package jPDFPreflightSamples;

import java.text.DateFormat;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Vector;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.tree.DefaultMutableTreeNode;

import com.qoppa.pdf.DocumentInfo;
import com.qoppa.pdfPreflight.profiles.Profile;
import com.qoppa.pdfPreflight.results.PreflightInfo;
import com.qoppa.pdfPreflight.results.PreflightResults;
import com.qoppa.pdfPreflight.results.ResultRecord;

public class PreflightResultsTreeFactory
{
	public final static Icon ERROR_ICON;
	public final static Icon INFO_ICON;
	public final static Icon CHECKMARK_ICON;
	public final static Icon WARN_ICON;
	public final static Icon COULD_FIX_ICON;

	static
	{
		ERROR_ICON = new ImageIcon(PreflightResultsTreeFactory.class.getResource("pferror.png"));
		INFO_ICON = new ImageIcon(PreflightResultsTreeFactory.class.getResource("pfinfo.png"));
		CHECKMARK_ICON = new ImageIcon(PreflightResultsTreeFactory.class.getResource("pfcheck.png"));
		WARN_ICON = new ImageIcon(PreflightResultsTreeFactory.class.getResource("pfwarn.png"));
		COULD_FIX_ICON = new ImageIcon(PreflightResultsTreeFactory.class.getResource("pfcouldfix.png"));
	}

	public static DefaultMutableTreeNode createResultsTree(PreflightResults results, Profile profile, boolean convertErrorsOnly, boolean canConvert, DocumentInfo docInfo)
	{
		Hashtable pageGroups = new Hashtable();
		TreeMap noPageGroups = new TreeMap();
		List resultList = results.getResults();
		for (int count = 0; count < resultList.size(); ++count)
		{
			ResultRecord rr = (ResultRecord) resultList.get(count);

			if (!convertErrorsOnly || (rr.isFixable() == false || rr.willFix() == false))
			{
				if (rr.getPageNumber() != -1)
				{
					insertIntoGroups(rr, pageGroups);
				}
				else
				{
					insertIntoGroups(rr, noPageGroups);
				}
			}
		}

		DefaultMutableTreeNode rootNode = new DefaultMutableTreeNode("Results");

		// Create results tree
		DefaultMutableTreeNode resultsNode = new DefaultMutableTreeNode("Preflight Results - " + profile.getName());
		addGroups(noPageGroups, resultsNode, convertErrorsOnly, canConvert);
		addGroups(pageGroups, resultsNode, convertErrorsOnly, canConvert);
		if (resultsNode.getChildCount() == 0)
		{
			resultsNode.add(new IconTreeNode("No problems found", CHECKMARK_ICON));
		}
		rootNode.add(resultsNode);

		// Create document overview tree
		rootNode.add(getOverviewNode(results, docInfo));

		// Create Preflight info tree
		rootNode.add(getPFInfoNode(results, profile));

		return rootNode;
	}

	private static DefaultMutableTreeNode getPFInfoNode(PreflightResults results, Profile profile)
	{
		PreflightInfo pi = results.getPFInfo();

		DefaultMutableTreeNode piNode = new DefaultMutableTreeNode("Preflight Information");
		piNode.add(new IconTreeNode("Profile: " + profile.getName(), INFO_ICON));
		piNode.add(new IconTreeNode("Version: " + pi.getVersion(), INFO_ICON));

		DateFormat dateFormat = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM);
		piNode.add(new IconTreeNode("Date: " + dateFormat.format(pi.getDateTime()), INFO_ICON));

		piNode.add(new IconTreeNode("User: " + pi.getUserName(), INFO_ICON));
		piNode.add(new IconTreeNode("Computer: " + pi.getComputerName(), INFO_ICON));
		piNode.add(new IconTreeNode("Oper System: " + pi.getOSInfo(), INFO_ICON));
		piNode.add(new IconTreeNode("Duration: " + formatDuration(pi.getDuration()), INFO_ICON));

		return piNode;
	}

	private static String formatDuration(long duration)
	{
		if (duration < 1000)
		{
			return duration + " ms";
		}
		else if (duration < 60000)
		{
			return duration / 1000 + " s";
		}
		else
		{
			return duration / 60000 + " mins";
		}
	}

	private static DefaultMutableTreeNode getOverviewNode(PreflightResults results, DocumentInfo docInfo)
	{
		DefaultMutableTreeNode overviewNode = new DefaultMutableTreeNode("Document Information");
		if (docInfo.getTitle() != null)
		{
			overviewNode.add(new IconTreeNode("Title: " + docInfo.getTitle(), INFO_ICON));
		}
		if (docInfo.getAuthor() != null)
		{
			overviewNode.add(new IconTreeNode("Author: " + docInfo.getAuthor(), INFO_ICON));
		}
		if (docInfo.getCreator() != null)
		{
			overviewNode.add(new IconTreeNode("Creator: " + docInfo.getCreator(), INFO_ICON));
		}
		if (docInfo.getProducer() != null)
		{
			overviewNode.add(new IconTreeNode("Producer: " + docInfo.getProducer(), INFO_ICON));
		}

		DateFormat dateFormat = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM);
		if (docInfo.getCreationDate() != null)
		{
			overviewNode.add(new IconTreeNode("Created: " + dateFormat.format(docInfo.getCreationDate()), INFO_ICON));
		}
		if (docInfo.getModDate() != null)
		{
			overviewNode.add(new IconTreeNode("Modified: " + dateFormat.format(docInfo.getModDate()), INFO_ICON));
		}

		overviewNode.add(new IconTreeNode("Page Count: " + docInfo.getPageCount(), INFO_ICON));

		return overviewNode;
	}

	private static void insertIntoGroups(ResultRecord rr, Map groups)
	{
		Vector groupList = (Vector) groups.get(rr.getHeader());
		if (groupList == null)
		{
			groupList = new Vector();
			groups.put(rr.getHeader(), groupList);
		}
		groupList.add(rr);
	}

	private static void addGroups(Map groups, DefaultMutableTreeNode parentNode, boolean convertErrorsOnly, boolean canConvert)
	{
		Iterator keys = groups.keySet().iterator();
		while (keys.hasNext())
		{
			String groupName = (String) keys.next();
			Vector resultList = (Vector) groups.get(groupName);
			if (resultList.size() == 1)
			{
				ResultRecord rr = (ResultRecord) resultList.get(0);
				if (!convertErrorsOnly || (rr.isFixable() == false || rr.willFix() == false))
				{
					if (canConvert && rr.isFixable() == true && rr.willFix() == false)
					{
						parentNode.add(new IconTreeNode(rr, COULD_FIX_ICON));
					}
					else if (canConvert && rr.isFixable() == true && rr.willFix() == true)
					{
						parentNode.add(new IconTreeNode(rr, WARN_ICON));
					}
					else
					{
						parentNode.add(new IconTreeNode(rr, ERROR_ICON));
					}
				}
			}
			else
			{
				String nodeLabel = groupName + ": " + resultList.size() + " times.";
				IconTreeNode groupNode = new IconTreeNode(nodeLabel, WARN_ICON);
				Icon worstLevel = WARN_ICON;
				int count = 0;
				for (int i = 0; i < resultList.size(); ++i)
				{
					ResultRecord rr = (ResultRecord) resultList.get(i);
					if (!convertErrorsOnly || (rr.isFixable() == false || rr.willFix() == false))
					{
						count++;
						if (canConvert && rr.isFixable() == true && rr.willFix() == false)
						{
							groupNode.add(new IconTreeNode(rr, COULD_FIX_ICON));
							if (worstLevel != ERROR_ICON)
							{
								worstLevel = COULD_FIX_ICON;
							}
						}
						else if (canConvert && rr.isFixable() == true && rr.willFix() == true)
						{
							groupNode.add(new IconTreeNode(rr, WARN_ICON));
						}
						else
						{
							groupNode.add(new IconTreeNode(rr, ERROR_ICON));
							worstLevel = ERROR_ICON;
						}
					}
				}
				groupNode.setIcon(worstLevel);
				nodeLabel = groupName + ": " + count + " times.";
				groupNode.setUserObject(nodeLabel);

				parentNode.add(groupNode);
			}
		}
	}
}